/* 
 * fread-fwrite.c 
 *
 * Копіює задану кількість байтів із одного файлу в другий.
 * Ілюструє порядок застосування функцій fopen(), fdopen(),
 * fread(), fwrite(), fclose().
 *
 */

#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>


/* Читає count байтів із файлу filename. 
   Повертає покажчик на буфер із прочитаними даними. */
char *read_from_file(const char *filename, size_t count)
{
        char *buf;
        FILE *stream;

        /* Розміщує буфер. */
        buf = malloc(count);
        if (buf == NULL) {
                /* Розмістити буфер не вдалось.
                   Виводить повідомлення про помилку */
                fprintf(stderr, "Not enough memory\n");
                /* і повертає управління. */
                return NULL;
        }

        /* Створює потік введення. */
        stream = fopen(filename, "r");
        if (stream == NULL)	{
                /* Створити потік не вдалось.
                   Виводить повідомлення про помилку, */
                fprintf(stderr, "Error opening input stream: %s\n",
                                                        strerror(errno));
                /* звільняє буфер */
                free(buf);
                /* і повертає управління. */
                return NULL;
        }

        /* Читає дані. */
        fread(buf, 1, count, stream);
        if (ferror(stream)) {
                /* Помилка читання.
                   Виводить повідомлення про помилку, */
                fprintf(stderr, "Error reading from input stream: %s\n",
                                                        strerror(errno));
                /* звільняє буфер, */
                free(buf);
                /* знищує потік введення */
                if (fclose(stream) == EOF) 
                        fprintf(stderr, "Error closing input stream: %s\n",
                                                        strerror(errno));
                /* і повертає управління. */
                return NULL;
        }

        /* Знищує потік введення. */
        if (fclose(stream) == EOF) 
                fprintf(stderr, "Error closing input stream: %s\n",
                                                        strerror(errno));
        /* Повертає управління. */
        return buf;
}

/* Записує count байтів із буфера buf у файл filename. */
size_t write_to_file(const char *filename, char *buf, size_t count)
{
        int fd;
        FILE *stream;
        size_t rcount;

        /* Відкриває файл для запису. */
        fd = open(filename, O_WRONLY | O_CREAT | O_EXCL, 00666);
        if (fd < 0) {
                /* Відкрити файл не вдалось.
                   Виводить повідомлення про помилку */
                fprintf(stderr, "Error opening output file: %s\n",
                                                        strerror(errno));
                /* і повертає управління. */
                return 0;
        }
        /* Створює потік виведення. */
        stream = fdopen(fd, "w");
        if (stream == NULL) {
                /* Створити потік виведення не вдалось.
                   Виводить повідомлення про помилку, */
                fprintf(stderr, "Error opening output stream: %s\n",
                                                        strerror(errno));
                /* закриває файл */
                if (close(fd) < 0)
                        fprintf(stderr, "Error closing output file: %s\n",
                                                        strerror(errno));
                /* і повертає управління. */
                return 0;
        }

        /* Записує дані. */
        rcount = fwrite(buf, 1, count, stream);
        if (ferror(stream)) {
                /* Помилка запису.
                   Виводить повідомлення про помилку, */
                fprintf(stderr, "Error writing to output stream: %s\n",
                                                        strerror(errno));
                /* знищує потік виведення */
                if (fclose(stream) == EOF) 
                        fprintf(stderr, "Error closing output"
                                        " stream: %s\n", strerror(errno));
                /* і повертає управління. */
                return 0;
        }

        /* Знищує потік виведення. */
        if (fclose(stream) == EOF) 
                fprintf(stderr, "Error closing output stream: %s\n",
                                                        strerror(errno));
        /* Повертає управління. */
        return rcount;
}

int main(int argc, char *argv[])
{
        char *src = argv[1];
        char *dst = argv[2];
        size_t count;
        char *buf;

        if (argc < 4) {
                /* Виводить повідомлення про порядок запуску. */
                fprintf(stderr, "Usage: %s source dest count\n",
                                                basename(argv[0]));
                exit(EXIT_FAILURE);
        }

        /* Перетворює розмір у числову форму. */
        count = (size_t) atoi(argv[3]);

        /* Читає дані з початкового файлу. */
        buf = read_from_file(src, count);
        if (buf == NULL) {
                /* Повідомляє про помилку читання і завершує роботу. */
                fprintf(stderr, "Can't read %d bytes from %s\n",
                                                        count, src);
                exit(EXIT_FAILURE);
        }
        /* Повідомляє про успіх читання. */
        printf("%d bytes from %s has been read\n", count, src);

        /* Записує дані у результуючий файл. */
        if (write_to_file(dst, buf, count) != count) {
                /* Повідомляє про помилку запису і завершує роботу. */
                fprintf(stderr, "Can't write %d bytes to %s\n",
                                                        count, dst);
                exit(EXIT_FAILURE);
        }
        /* Повідомляє про успіх запису. */
        printf("%d bytes to %s has been written\n", count, dst);

        /* Звільняє буфер. */
        /* Потрібно робити тільки тоді, коли ще повинна виконуватись 
           якась робота (а не відразу exit()). */
        free(buf);                  
        /* Завершує роботу. */
        exit(EXIT_SUCCESS);
}
